%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Demonstration of the Variational Bayesian Adaptive Kalman Filter
% with simple linear resonator model.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Copyright (C) 2006 Simo Srkk
%
% $Id: vbakf_demo1.m,v 1.9 2007/08/21 20:06:33 ssarkka Exp $
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

  randn('state',0);

  %
  % Model is
  %
  %   dx1/dt = w1(t)
  %  dx2/dt = - 0.05 x2 + w2(t),
  %      y_k = x1(k*dt) + x2(k*dt) + v_k
  %
  % where w(t) is a white noise process and
  % v_k is a Gaussian random variable.
  %
  dt = 0.1;
  m0 = [0;0;0.1];
  P0 = eye(3);
  F = [0 0 0;
       0 0 1;
       0 -1/50 0];
  L = [1 0;
       0 0;
       0 1];
  H = [1 1 0];
  Qc = diag([0.1 0.001]);
  
  [A,Q] = lti_disc(F,L,Qc,dt);
  
  %
  % Simulate
  %
  steps = round(400/dt);
  x = gauss_rnd(m0,P0);
  t = 0;

  X = zeros(size(m0,1),steps);
  Y = zeros(1,steps);
  V = zeros(1,steps);
  T = zeros(1,steps);
  for k=1:steps
    if k<1.5*steps/4
      r = 0.2 + 0.4*(1+tanh(dt*0.1*(k-steps/4)));
    else
      r = 0.2 + 0.4*(1+tanh(-dt*0.1*(k-2*steps/4)));
    end
    x = gauss_rnd(A*x,Q);
    y = gauss_rnd(H*x,r);
    t = t + dt;
    X(:,k) = x;
    Y(:,k) = y;
    V(k)   = r;
    T(k)   = t;    
  end

  clf;
  subplot(1,2,1);
  plot(T,H*X,T,Y,'o');
  title('signal');
  legend('x1+x2','y');

  subplot(1,2,2);
  plot(T,V,'-');
  axis([0 T(end) 0 2*max(V)]);
  title('vbakf variance');
  legend('True','Estimated mean');
  
  fprintf('The data. press enter.\n');
  pause;

  vb_m = m0;
  vb_P = P0;
  vb_a = 1;
  vb_b = 1;
  
  akf_MM = zeros(size(m0,1),size(Y,2));
  akf_VV = zeros(1,size(Y,2));

  akf_Q_05 = zeros(1,size(Y,2));
  akf_Q_95 = zeros(1,size(Y,2));
  
  clf;
  rho = 1-exp(-4);
  for k=1:size(Y,2)
    [vb_m,vb_P,vb_a,vb_b] = vbakf_predict_1d(vb_m,vb_P,A,Q,...
                                             vb_a,vb_b,rho);

    [vb_m,vb_P,vb_a,vb_b] = vbakf_update_1d(vb_m,vb_P,...
                                            Y(:,k),H, ...
                                            vb_a,vb_b);
    
    akf_MM(:,k) = vb_m;
    akf_VV(k) = vb_b/(vb_a-1);
      
    theta = (0.05:0.01:2);
    p = -(vb_a+1)*log(theta) - vb_b./theta;
    p = p - median(p);
    p = exp(p);
    p = p./sum(p);

    c = cumsum(p);
    akf_Q_05(k) = theta(max([find(c < 0.05) 1]));
    akf_Q_95(k) = theta(min([find(c > 0.95) length(c)]));


    %
    % Plot
    %
    if rem(k,50) == 0
      subplot(1,3,1);

      r = V(k);
      plot(theta,p,[r r],[0 1.1*max(p)]);
      title('vbakf p(\theta)');
      
      subplot(1,3,2);
      plot(T,H*X,'g--',T,Y,'.',T(1:k),H*akf_MM(:,1:k),'k-');
      title('vbakf estimate');

      subplot(1,3,3);
      plot(T,V,'--',T(1:k),akf_VV(1:k),'-',...
           T(1:k),akf_Q_05(1:k),':',T(1:k),akf_Q_95(1:k),':');
      axis([0 T(end) 0 2*max(V)]);
      title('vbakf variance');
      legend('True','Estimated mean');
      
      drawnow;
      pause(0.01);
    end
  end
  
  rmse_akf = mean(sum((X-akf_MM).^2,1));
  fprintf('AKF-RMSE = %.4f\n',rmse_akf);
  
  
  %
  % Run IMM
  %
  imm_MM = zeros(size(m0,1),size(Y,2));
  imm_VV = zeros(1,size(Y,2));
  imm_Q_05 = zeros(1,size(Y,2));
  imm_Q_95 = zeros(1,size(Y,2));
  
  RR = 0.1:0.01:1.2;
  RR = reshape(RR,[1 1 length(RR)]);
  imm_M = repmat(m0,1,length(RR));
  imm_P = repmat(P0,[1 1 length(RR)]);
  W = ones(1,length(RR)) / length(RR);

  lambda = 5;
  for i=1:length(RR)
    for j=1:length(RR)
      TP(i,j) = exp(-lambda*dt*abs(i-j));
    end
  end
  for j=1:size(TP,2)
    TP(:,j) = TP(:,j) / sum(TP(:,j));
  end
  
  clf;
  for k=1:size(Y,2)
    [imm_M,imm_P,W] = imm_predict(imm_M,imm_P,W,TP,A,Q);
    [imm_M,imm_P,W] = imm_update(imm_M,imm_P,W,Y(:,k),H,RR);

    [M,P] = imm_estimate(imm_M,imm_P,W);
    
    theta = reshape(RR,1,length(RR));
    imm_MM(:,k) = M;
    imm_VV(k) = sum(W .* theta);

    c = cumsum(W);
    imm_Q_05(k) = theta(max([find(c < 0.05) 1]));
    imm_Q_95(k) = theta(min([find(c > 0.95) length(c)]));

    %
    % Plot
    %
    if rem(k,50) == 0
      subplot(1,3,1);

      r = V(k);
      plot(theta,W,[r r],[0 1.1*max(W)]);
      title('imm p(\theta)');
      
      subplot(1,3,2);
      plot(T,H*X,'g--',T,Y,'.',T(1:k),H*imm_MM(:,1:k),'k-');
      title('vbakf estimate');

      subplot(1,3,3);
      plot(T,V,'--',T(1:k),imm_VV(1:k),'-',...
           T(1:k),imm_Q_05(1:k),':',T(1:k),imm_Q_95(1:k),':');
      axis([0 T(end) 0 2*max(V)]);
      title('imm variance');
      legend('True','Estimated mean');
      
      drawnow;
      pause(0.01);
    end
  end

  rmse_imm = mean(sum((X-imm_MM).^2,1));
  fprintf('IMM-RMSE = %.4f\n',rmse_imm);
  
  %
  % Run KF's with different noises
  %
  RR = 0.1:0.01:1.2;
  rmse_kf = [];
  for i=1:length(RR)
    MM = zeros(size(m0,1),size(Y,2));
    m = m0;
    P = P0;
    
    for k=1:size(Y,2)
      [m,P] = kf_predict(m,P,A,Q);
      [m,P] = kf_update(m,P,Y(:,k),H,RR(i));
      MM(:,k) = m;
    end

    rmse_kf(i) = mean(sum((X-MM).^2,1));
    fprintf('KF(%.2f)-RMSE = %.4f\n',RR(i),rmse_kf(i));
  end
  
