%KF_SMOOTH  Kalman Smoother
%
% Syntax:
%   [X,P,S] = KF_SMOOTH(X,P,A,Q)
%
% Author:
%   Simo Srkk, 2003
%
% In:
%   X - NxM matrix of M mean estimates from Kalman filter
%   P - NxNxM matrix of M state covariances from Kalman Filter
%   A - NxN state transition matrix or NxNxM matrix of M state
%       transition matrices for each step.
%   Q - NxN process noise covariance matrix or NxNxM matrix
%       of M state process noise covariance matrices for each step.
%
% Out:
%   X - Smoothed state mean sequence
%   P - Smoothed state covariance sequence
%   S - Smoother gain sequence
%   
% Description:
%   Kalman smoother algorithm. Calculate "smoothed"
%   sequence from given Kalman filter output sequence
%   by conditioning all steps to all measurements.
%
% Example:
%   m = m0;
%   P = P0;
%   MM = zeros(size(m,1),size(Y,2));
%   PP = zeros(size(m,1),size(m,1),size(Y,2));
%   for k=1:size(Y,2)
%     [m,P] = kf_predict(m,P,A,Q);
%     [m,P] = kf_update(m,P,Y(:,k),H,R);
%     MM(:,k) = m;
%     PP(:,:,k) = P;
%   end
%   [MM,PP] = kf_smooth(MM,PP,A);
%
% See also:
%   KF_PREDICT, KF_UPDATE

% History:
%   23.02.2003  The first version.
%
% Copyright (C) 2003 Simo Srkk
%
% $Id: kf_smooth.m,v 1.1.1.1 2006/03/28 20:10:59 ssarkka Exp $
%
% This software is distributed under the GNU General Public 
% Licence (version 2 or later); please refer to the file 
% Licence.txt, included with the software, for details.

function [X,P,S] = kf_smooth(X,P,A,Q)

  %
  % Check which arguments are there
  %
  if nargin < 4
    error('Too few arguments');
  end

  %
  % Extend A and Q if they are NxN matrices
  %
  if size(A,3)==1
    A = repmat(A,[1 1 size(X,2)]);
  end
  if size(Q,3)==1
    Q = repmat(Q,[1 1 size(X,2)]);
  end

  %
  % Run the smoother
  %
  S = zeros(size(X,1),size(X,1),size(X,2));
  for k=(size(X,2)-1):-1:1
    P_pred   = A(:,:,k) * P(:,:,k) * A(:,:,k)' + Q(:,:,k);
    S(:,:,k) = P(:,:,k) * A(:,:,k)' / P_pred;
    X(:,k)   = X(:,k) + S(:,:,k) * (X(:,k+1) - A(:,:,k) * X(:,k));
    P(:,:,k) = P(:,:,k) + S(:,:,k) * (P(:,:,k+1) - P_pred) * S(:,:,k)';
  end

